import numpy as np
import matplotlib.pyplot as plt

# Parameters
N = 1000  # Number of cells
n_trials = 150  # Number of trials
p_flip_1_to_0 = 0.01  # Probability of flipping from 1 to 0
p_flip_0_to_1 = 0.01  # Probability of flipping from 0 to 1

# Initialize an array to track the number of flips for each cell
flip_counts = np.zeros(N, dtype=int)
# Initialize arrays to track the proportions of cells in each category at each trial
proportions_over_trials = np.zeros((n_trials + 1, 5), dtype=float)  # 0, 1, 2, 3, 4+
average_values = np.zeros(n_trials + 1)  # Array to store average values

# Count initial state (all cells are at 0 flips)
proportions_over_trials[0, 0] = 1.0  # All cells start at 1 (100%)
average_values[0] = 1.0  # All cells start at 1

# Simulate the flipping process
for trial in range(n_trials):
    for i in range(N):
        rand_num = np.random.rand()
        
        if flip_counts[i] % 2 == 0:  # Currently in state 1
            if rand_num < p_flip_1_to_0:
                flip_counts[i] += 1  # Flip to 0
        else:  # Currently in state 0
            if rand_num < p_flip_0_to_1:
                flip_counts[i] += 1  # Flip back to 1

    # Count the number of cells in each flip category after this trial
    for flips in flip_counts:
        if flips < 5:
            proportions_over_trials[trial + 1, flips] += 1
        else:
            proportions_over_trials[trial + 1, 4] += 1  # Count all > 4 as "4 or more"

    # Calculate the proportions by dividing by the total number of cells
    proportions_over_trials[trial + 1] /= N

    # Calculate the average value of all cells
    average_values[trial + 1] = np.mean(1 - (flip_counts % 2))  # Average value of cells

# Prepare data for plotting
trials = np.arange(n_trials + 1)

# Figure dimensions and font size
fig_width, fig_height = 5, 3  # inches
font_size = 14

# Plotting
plt.figure(figsize=(fig_width, fig_height))
for i in range(5):
    plt.plot(trials, proportions_over_trials[:, i], label=f'{i} Flips' if i < 4 else '4+ Flips')

# Plot average values
plt.plot(trials, average_values, label='Average Value', color='black', linestyle='--', linewidth=2)

# Adding labels and title
plt.title('Proportion of Cells by Flip Count and Average Value Over Trials', fontsize=font_size)
plt.xlabel('Trial Number', fontsize=font_size)
plt.ylabel('Proportion of Cells / Average Value', fontsize=font_size)
plt.legend(loc='center left', bbox_to_anchor=(1, 0.5), fontsize=font_size)  # Legend to the right
plt.ylim(0, 1)  # Set Y-axis limit to range from 0 to 1
plt.grid(axis='y')

# Save the figure as a JPEG file with 1200 DPI
plt.savefig('cell_flip_simulation.jpg', dpi=1200, bbox_inches='tight')

# Show the plot
plt.show()

